﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/discovery/ApplicationDiscoveryServiceRequest.h>
#include <aws/discovery/ApplicationDiscoveryService_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ApplicationDiscoveryService {
namespace Model {

/**
 */
class StartImportTaskRequest : public ApplicationDiscoveryServiceRequest {
 public:
  AWS_APPLICATIONDISCOVERYSERVICE_API StartImportTaskRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartImportTask"; }

  AWS_APPLICATIONDISCOVERYSERVICE_API Aws::String SerializePayload() const override;

  AWS_APPLICATIONDISCOVERYSERVICE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Optional. A unique token that you can provide to prevent the same import
   * request from occurring more than once. If you don't provide a token, a token is
   * automatically generated.</p> <p>Sending more than one
   * <code>StartImportTask</code> request with the same client request token will
   * return information about the original import task with that client request
   * token.</p>
   */
  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  StartImportTaskRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A descriptive name for this request. You can use this name to filter future
   * requests related to this import task, such as identifying applications and
   * servers that were included in this import task. We recommend that you use a
   * meaningful name for each import task.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  StartImportTaskRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The URL for your import file that you've uploaded to Amazon S3.</p>
   * <p>If you're using the Amazon Web Services CLI, this URL is structured as
   * follows: <code>s3://BucketName/ImportFileName.CSV</code> </p>
   */
  inline const Aws::String& GetImportUrl() const { return m_importUrl; }
  inline bool ImportUrlHasBeenSet() const { return m_importUrlHasBeenSet; }
  template <typename ImportUrlT = Aws::String>
  void SetImportUrl(ImportUrlT&& value) {
    m_importUrlHasBeenSet = true;
    m_importUrl = std::forward<ImportUrlT>(value);
  }
  template <typename ImportUrlT = Aws::String>
  StartImportTaskRequest& WithImportUrl(ImportUrlT&& value) {
    SetImportUrl(std::forward<ImportUrlT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::String m_name;

  Aws::String m_importUrl;
  bool m_clientRequestTokenHasBeenSet = true;
  bool m_nameHasBeenSet = false;
  bool m_importUrlHasBeenSet = false;
};

}  // namespace Model
}  // namespace ApplicationDiscoveryService
}  // namespace Aws
