!===============================================================================
! Copyright (C) 2025 Intel Corporation
!
! This software and the related documents are Intel copyrighted  materials,  and
! your use of  them is  governed by the  express license  under which  they were
! provided to you (License).  Unless the License provides otherwise, you may not
! use, modify, copy, publish, distribute,  disclose or transmit this software or
! the related documents without Intel's prior written permission.
!
! This software and the related documents  are provided as  is,  with no express
! or implied  warranties,  other  than those  that are  expressly stated  in the
! License.
!===============================================================================

!
!    Content : Intel(R) oneAPI Math Kernel Library (oneMKL)
!              Inspector-Executor Sparse BLAS Fortran example
!              for the mkl_sparse_?_dense2bsr routine to demonstrate
!              conversion from a dense matrix to a block compressed
!              sparse row (BSR) sparse matrix format.
!
!===============================================================================
!
! Example program for using Intel(R) oneMKL Inspector-Executor Sparse BLAS
! routines for conversions from dense to sparse matrix representation.
!
! The following Inspector-Executor Sparse BLAS routines are used in the example:
!
!   Initialization/Destruction stage:
!          mkl_sparse_destroy
!
!   Format conversion function:
!          mkl_sparse_d_dense2bsr
!
!   Export function for sparse data arrays:
!          mkl_sparse_d_export_bsr
!
! Consider the matrix A below to be represented in block compressed sparse row
! (BSR) format (see 'Sparse Matrix Storage Schemes' in the Intel(R) oneMKL
! Fortran Reference Manual):
!
!       |  1  -2   0   0 |
!       |  3  -4   0   0 |
!   A = |  0   0   5  -6 |.
!       |  0   0   7  -8 |
!       |  9 -10   0   0 |
!       | 11 -12   0   0 |
!
!
! A BSR representation of the matrix with column-major layout, 1-based indexing,
! block size 2 (i.e. with square blocks of dimension 2x2), and three arrays is:
!
!     bsrNrows       = 3
!     bsrNcols       = 2
!     bsrBlockLayout = SPARSE_LAYOUT_COLUMN_MAJOR
!     bsrBlockSize   = 2
!     bsrNnz         = 3
!     bsrIndex       = SPARSE_INDEX_BASE_ONE
!     bsrRowPtr      = (1               2               3               4)
!     bsrColIdx      = (1               2               1)
!     bsrValues      = (1   3  -2  -4   5   7  -6  -8   9  11 -10 -12)
!
! This example presents:
!    * mkl_sparse_d_dense2bsr() usage to convert a dense matrix to
!      a sparse matrix in BSR format
!    * manual computation of a reference dense matrix-vector product from the
!      original dense matrix
!    * usage of mkl_sparse_d_mv() for the computation of a sparse matrix-vector
!      product using the sparse matrix in BSR format
!    * comparison and validation of the resulting vectors
!
!
! Note: The I/E Sparse BLAS solution supports conversion from a dense matrix to
!       any of the following sparse formats:
!       - Compressed Sparse Row (CSR)
!       - Compressed Sparse Column (CSC)
!       - Block Compressed Sparse Row (BSR)
!       - Coordinate (COO)
!       This example demonstrates conversion to BSR format using the
!       mkl_sparse_?_dense2bsr() API. To convert to other sparse formats,
!       follow a similar approach and use the corresponding API:
!       - mkl_sparse_?_dense2csr()
!       - mkl_sparse_?_dense2csc()
!       - mkl_sparse_?_dense2coo()
!
!===============================================================================
!

SUBROUTINE print_int_value(name, val)
    IMPLICIT NONE
    CHARACTER(LEN=*), INTENT(IN) :: name
    INTEGER, INTENT(IN) :: val
    PRINT *, TRIM(ADJUSTL(name)), " = ", val
END SUBROUTINE print_int_value

SUBROUTINE print_int_array(name, array, len)
    IMPLICIT NONE
    CHARACTER(LEN=*), INTENT(IN) :: name
    INTEGER, INTENT(IN), DIMENSION(*) :: array
    INTEGER, INTENT(IN) :: len
    INTEGER :: i
    WRITE(*, '(A, A, A)', ADVANCE="NO") " ", TRIM(ADJUSTL(name)), " ="
    DO i = 1, len
        WRITE(*,  '(I3)', ADVANCE="NO") array(i)
        IF (i < len) WRITE(*, '(A)', ADVANCE="NO") ", "
    END DO
    PRINT *
END SUBROUTINE print_int_array

SUBROUTINE print_index(name, idxBase)
    USE MKL_SPBLAS
    IMPLICIT NONE
    CHARACTER(LEN=*), INTENT(IN) :: name
    INTEGER(C_INT), INTENT(IN) :: idxBase
    CHARACTER(LEN=25) :: idxBaseStr
    IF (idxBase == SPARSE_INDEX_BASE_ZERO) THEN
        idxBaseStr = "SPARSE_INDEX_BASE_ZERO"
    ELSE
        idxBaseStr = "SPARSE_INDEX_BASE_ONE"
    END IF
    PRINT *, TRIM(ADJUSTL(name)), " = ", TRIM(idxBaseStr)
END SUBROUTINE print_index

SUBROUTINE print_flt_array(name, array, len)
    IMPLICIT NONE
    CHARACTER(LEN=*), INTENT(IN) :: name
    DOUBLE PRECISION, INTENT(IN), DIMENSION(*) :: array
    INTEGER, INTENT(IN) :: len
    INTEGER :: i
    WRITE(*, '(A, A, A)', ADVANCE="NO") " ", TRIM(ADJUSTL(name)), " ="
    DO i = 1, len
        WRITE(*,  '(F4.0)', ADVANCE="NO") array(i)
        IF (i < len) WRITE(*, '(A)', ADVANCE="NO") ", "
    END DO
    PRINT *
END SUBROUTINE print_flt_array

SUBROUTINE print_layout(name, layout)
    USE MKL_SPBLAS
    IMPLICIT NONE
    CHARACTER(LEN=*), INTENT(IN) :: name
    INTEGER(C_INT), INTENT(IN) :: layout
    CHARACTER(LEN=30) :: layoutStr
    IF (layout == SPARSE_LAYOUT_ROW_MAJOR) THEN
        layoutStr = "SPARSE_LAYOUT_ROW_MAJOR"
    ELSE
        layoutStr = "SPARSE_LAYOUT_COLUMN_MAJOR"
    END IF
    PRINT *, TRIM(ADJUSTL(name)), " = ", TRIM(layoutStr)
END SUBROUTINE print_layout

PROGRAM sparse_dense2bsr_example
    USE MKL_SPBLAS
    USE ISO_C_BINDING
    IMPLICIT NONE

    INTEGER, PARAMETER :: M = 6
    INTEGER, PARAMETER :: N = 4
    INTEGER, PARAMETER :: bsrBlockSize = 2
    INTEGER :: exit_status

    INTEGER(C_INT) :: dense_layout
    INTEGER(C_INT) :: bsrBlockLayout
    INTEGER(C_INT) :: indexing
    TYPE(SPARSE_MATRIX_T) :: bsrA
    TYPE(MATRIX_DESCR) :: descrA
    INTEGER(C_INT) :: info
    INTEGER(C_INT) :: i, j

    DOUBLE PRECISION :: denseMatrix(M, N)
    DOUBLE PRECISION :: x(N)
    DOUBLE PRECISION :: y_ref(M)
    DOUBLE PRECISION :: y(M)

    ! BSR export variables
    INTEGER(C_INT) :: indexingOut, bsrBlockLayoutOut
    INTEGER :: bsrBlockNrows, bsrBlockNcols, bsrBlockSizeOut
    TYPE(C_PTR)      :: bsrRowStart_c   , bsrRowEnd_c   , bsrColIdx_c
    INTEGER, POINTER :: bsrRowStart_f(:), bsrRowEnd_f(:), bsrColIdx_f(:)
    TYPE(C_PTR)               :: bsrValues_c
    DOUBLE PRECISION, POINTER :: bsrValues_f(:)
    INTEGER :: index, bsrBlockNnz, bsrNnz
    DOUBLE PRECISION :: TOL
    INTEGER :: conversion_passed

    !*******************************************************************************
    !     Declaration and initialization of parameters for the dense representation
    !     of the matrix A
    !*******************************************************************************
    dense_layout = SPARSE_LAYOUT_COLUMN_MAJOR
    bsrBlockLayout = SPARSE_LAYOUT_COLUMN_MAJOR
    indexing = SPARSE_INDEX_BASE_ONE
    descrA%type = SPARSE_MATRIX_TYPE_GENERAL

    denseMatrix = RESHAPE([ &
         1.0D0,  3.0D0,  0.0D0,  0.0D0,  9.0D0, 11.0D0, &
        -2.0D0, -4.0D0,  0.0D0,  0.0D0,-10.0D0,-12.0D0, &
         0.0D0,  0.0D0,  5.0D0,  7.0D0,  0.0D0,  0.0D0, &
         0.0D0,  0.0D0, -6.0D0, -8.0D0,  0.0D0,  0.0D0], [M,N])

    x = 1.0D0
    y_ref = 0.0D0
    y = 0.0D0

    PRINT *, ""
    PRINT *, "Example program for conversion from a dense matrix to"
    PRINT *, " a sparse matrix in BSR format using IE Sparse BLAS APIs"
    PRINT *, "-------------------------------------------------------------------------"

    !******************************************************************************
    !    Input dense matrix
    !******************************************************************************
    PRINT *, "[Input] Matrix array in dense format:"
    CALL print_int_value("nrows", M)
    CALL print_int_value("ncols", N)
    PRINT *, "    denseMatrix:"
    DO i = 1, M
        WRITE(*, "(A)", ADVANCE="NO") "    "
        DO j = 1, N
            WRITE(*, "(F6.0)", ADVANCE="NO") denseMatrix(i, j)
        END DO
        PRINT *
    END DO
    PRINT *, ""

    !******************************************************************************
    !    Convert from DENSE to BSR format
    !******************************************************************************
    exit_status = 0
    info = mkl_sparse_d_dense2bsr(M, N, dense_layout, M, denseMatrix, &
                                  indexing, bsrBlockSize, bsrBlockLayout, descrA, bsrA)
    IF (info /= SPARSE_STATUS_SUCCESS) THEN
        PRINT *, " Error in mkl_sparse_d_dense2bsr: ", info
        exit_status = 1
        GOTO 999
    END IF

    !******************************************************************************
    !    Export and print the BSR matrix
    !******************************************************************************
    info = mkl_sparse_d_export_bsr(bsrA, indexingOut, bsrBlockLayoutOut, bsrBlockNrows, &
                                   bsrBlockNcols, bsrBlockSizeOut, bsrRowStart_c, &
                                   bsrRowEnd_c, bsrColIdx_c, bsrValues_c)
    IF (info /= SPARSE_STATUS_SUCCESS) THEN
        PRINT *, " Error in mkl_sparse_d_export_bsr: ", info
        exit_status = 1
        GOTO 999
    END IF

    IF (bsrBlockLayoutOut /= bsrBlockLayout) THEN
        PRINT *, " Error: blockLayoutOut does not match input bsrBlockLayout"
        exit_status = 1
        GOTO 999
    END IF
    IF (bsrBlockSizeOut /= bsrBlockSize) THEN
        PRINT *, " Error: blockSizeOut does not match input bsrBlockSize"
        exit_status = 1
        GOTO 999
    END IF
    IF (indexingOut /= indexing) THEN
        PRINT *, " Error: indexingOut does not match input indexing"
        exit_status = 1
        GOTO 999
    END IF

    !   Converting C into Fortran pointers
    call C_F_POINTER(bsrRowStart_c, bsrRowStart_f, [bsrBlockNrows+1])
    index = 0
    IF (indexing == SPARSE_INDEX_BASE_ONE) index = 1
    bsrBlockNnz = bsrRowStart_f(bsrBlockNrows+1) - index
    bsrNnz = bsrBlockNnz * bsrBlockSize * bsrBlockSize
    call C_F_POINTER(bsrColIdx_c, bsrColIdx_f, [bsrBlockNnz])
    call C_F_POINTER(bsrValues_c, bsrValues_f, [bsrNnz])

    PRINT *, "[Output] Matrix in BSR format:"
    CALL print_int_value("bsrBlockNrows", bsrBlockNrows)
    CALL print_int_value("bsrBlockNcols", bsrBlockNcols)
    CALL print_int_value("bsrBlockSize", bsrBlockSize)
    CALL print_layout("bsrBlockLayout", bsrBlockLayout)
    CALL print_index("bsrIndexing", indexing)
    CALL print_int_value("bsrBlockNnz", bsrBlockNnz)
    CALL print_int_value("bsrNnz", bsrNnz)
    CALL print_int_array("bsrRowPtr", bsrRowStart_f, bsrBlockNrows + 1)
    CALL print_int_array("bsrColIdx", bsrColIdx_f, bsrBlockNnz)
    CALL print_flt_array("bsrValues", bsrValues_f, bsrNnz)
    PRINT *, ""

    !******************************************************************************
    !    Compute the reference matrix-vector solution from the dense matrix
    !******************************************************************************
    DO i = 1, M
        y_ref(i) = 0.0D0
        DO j = 1, N
            y_ref(i) = y_ref(i) + denseMatrix(i, j) * x(j)
        END DO
    END DO

    !******************************************************************************
    !    Compute the matrix-vector solution using the sparse matrix in BSR format
    !******************************************************************************
    info = mkl_sparse_d_mv(SPARSE_OPERATION_NON_TRANSPOSE, 1.0D0, bsrA, descrA, x, 0.0D0, y)
    IF (info /= SPARSE_STATUS_SUCCESS) THEN
        PRINT *, " Error in mkl_sparse_d_mv: ", info
        exit_status = 1
        GOTO 999
    END IF

    !******************************************************************************
    !    Validate the resulting vector y against the reference solution y_ref
    !******************************************************************************
    conversion_passed = 1
    TOL = 1.0D-14
    DO i = 1, M
        IF (ABS(y(i) - y_ref(i)) > TOL) THEN
            conversion_passed = 0
            exit_status = 1
            EXIT
        END IF
    END DO

    IF (conversion_passed == 1) THEN
        PRINT *, " The conversion from dense representation to BSR format passed"
    ELSE
        PRINT *, " The conversion from dense representation to BSR format failed"
        GOTO 999
    END IF
    PRINT *, ""

999 CONTINUE
    ! Release matrix handle
    info = mkl_sparse_destroy(bsrA)
    IF (info /= SPARSE_STATUS_SUCCESS) THEN
        PRINT *, " Error in mkl_sparse_destroy: ", info
        exit_status = 1
    END IF

    call exit(exit_status)
END PROGRAM sparse_dense2bsr_example
