/*******************************************************************************
 * Copyright 2005 Intel Corporation.
 *
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them is governed by
 * the express license under which they were provided to you ('License'). Unless the License provides otherwise,
 * you may not use, modify, copy, publish, distribute, disclose or transmit this software or the related
 * documents without Intel's prior written permission.
 * This software and the related documents are provided as is, with no express or implied warranties, other than
 * those that are expressly stated in the License.
 *******************************************************************************/

#include "base_image.h"

#pragma pack(push, 2)
struct BMPFileHeader {
    unsigned short bfType;
    unsigned int bfSize;
    unsigned short bfReserved1;
    unsigned short bfReserved2;
    unsigned int bfOffBits;
};

struct BMPCoreHeader {
    unsigned int bcSize;
    unsigned short bcWidth;
    unsigned short bcHeight;
    unsigned short bcPlanes;
    unsigned short bcBitCount;
};

struct BMPInfoHeader {
    unsigned int biSize;
    int biWidth;
    int biHeight;
    unsigned short biPlanes;
    unsigned short biBitCount;
    unsigned int biCompression;
    unsigned int biSizeImage;
    int biXPelsPerMeter;
    int biYPelsPerMeter;
    unsigned int biClrUsed;
    unsigned int biClrImportant;
};

struct __CIEXYZ {
    unsigned int ciexyzX;
    unsigned int ciexyzY;
    unsigned int ciexyzZ;
};
struct __CIEXYZTRIPLE {
    __CIEXYZ ciexyzRed;
    __CIEXYZ ciexyzGreen;
    __CIEXYZ ciexyzBlue;
};
struct BMPV4Header {
    unsigned int bV4RedMask;
    unsigned int bV4GreenMaskc;
    unsigned int bV4BlueMask;
    unsigned int bV4AlphaMask;
    unsigned int bV4CSType;
    __CIEXYZTRIPLE bV4Endpoints;
    unsigned int bV4GammaRed;
    unsigned int bV4GammaGreen;
    unsigned int bV4GammaBlue;
};

struct BMPV5Header {
    unsigned int bV5Intent;
    unsigned int bV5ProfileData;
    unsigned int bV5ProfileSize;
    unsigned int bV5Reserved;
};
#pragma pack(pop)

struct RGBquad {
    unsigned char rgbBlue;
    unsigned char rgbGreen;
    unsigned char rgbRed;
    unsigned char rgbReserved;
};

template <class T> static Status cppiSwapChannesARGB(void *pSrc, size_t srcStep, void *pDst, size_t dstStep, Size size)
{
    T iTemp;

    for (long long i = 0; i < size.height; i++) {
        T *pSrcRow = (T *)((char *)pSrc + i * srcStep);
        T *pDstRow = (T *)((char *)pDst + i * dstStep);

        for (long long j = 0, k = 0; j < size.width; j++, k += 4) {
            iTemp = pSrcRow[k];
            pDstRow[k] = pSrcRow[k + 1];
            pDstRow[k + 1] = pSrcRow[k + 2];
            pDstRow[k + 2] = pSrcRow[k + 3];
            pDstRow[k + 3] = iTemp;
        }
    }

    return STS_OK;
}

#define OWN_SAFE_READ(POINTER, BYTE_COUNT)                     \
    if ((BYTE_COUNT) != file.Read((POINTER), 1, (BYTE_COUNT))) \
    return STS_ERR_NOT_ENOUGH_DATA
#define OWN_SAFE_WRITE(POINTER, BYTE_COUNT)                     \
    if ((BYTE_COUNT) != file.Write((POINTER), 1, (BYTE_COUNT))) \
    return STS_ERR_FAILED

Status BmpReadHeader(File &file, BMPInfoHeader *pHeader, Image &image)
{
    BMPFileHeader fHeader;
    BMPInfoHeader header;

    if (!pHeader)
        pHeader = &header;
    memset(pHeader, 0, sizeof(BMPInfoHeader));

    // read header
    OWN_SAFE_READ(&fHeader.bfType, sizeof(fHeader.bfType));
    OWN_SAFE_READ(&fHeader.bfSize, sizeof(fHeader.bfSize));

    if (fHeader.bfType != 0x4D42)
        return STS_ERR_FORMAT;

    OWN_SAFE_READ(&fHeader.bfReserved1, sizeof(fHeader.bfReserved1));
    OWN_SAFE_READ(&fHeader.bfReserved2, sizeof(fHeader.bfReserved2));
    OWN_SAFE_READ(&fHeader.bfOffBits, sizeof(fHeader.bfOffBits));

    OWN_SAFE_READ(&pHeader->biSize, sizeof(pHeader->biSize));

    if (pHeader->biSize == sizeof(BMPCoreHeader)) {
        BMPCoreHeader coreHeader;
        coreHeader.bcSize = pHeader->biSize;
        OWN_SAFE_READ(&coreHeader.bcWidth, sizeof(coreHeader.bcWidth));
        OWN_SAFE_READ(&coreHeader.bcHeight, sizeof(coreHeader.bcHeight));
        OWN_SAFE_READ(&coreHeader.bcPlanes, sizeof(coreHeader.bcPlanes));
        OWN_SAFE_READ(&coreHeader.bcBitCount, sizeof(coreHeader.bcBitCount));

        pHeader->biWidth = (int)coreHeader.bcWidth;
        pHeader->biHeight = (int)coreHeader.bcHeight;
        pHeader->biPlanes = coreHeader.bcPlanes;
        pHeader->biBitCount = coreHeader.bcBitCount;
    } else {
        if (pHeader->biSize != sizeof(BMPInfoHeader) && pHeader->biSize != (sizeof(BMPV4Header) + sizeof(BMPInfoHeader)) &&
            pHeader->biSize != (sizeof(BMPV5Header) + sizeof(BMPV4Header) + sizeof(BMPInfoHeader)))
            return STS_ERR_UNSUPPORTED;

        OWN_SAFE_READ(&pHeader->biWidth, sizeof(pHeader->biWidth));
        OWN_SAFE_READ(&pHeader->biHeight, sizeof(pHeader->biHeight));
        OWN_SAFE_READ(&pHeader->biPlanes, sizeof(pHeader->biPlanes));
        OWN_SAFE_READ(&pHeader->biBitCount, sizeof(pHeader->biBitCount));
        OWN_SAFE_READ(&pHeader->biCompression, sizeof(pHeader->biCompression));

        if (pHeader->biWidth < 0)
            return STS_ERR_FAILED;
        if (pHeader->biBitCount != 8 && pHeader->biBitCount != 24 && pHeader->biBitCount != 32)
            return STS_ERR_FAILED;

        switch (pHeader->biCompression) {
        case 0L: // 0L == BI_RGB
            break;

        case 3L: // 3L == BI_BITFIELDS (we support only 8uC4 images)
        {
            if (pHeader->biBitCount != 32)
                return STS_ERR_UNSUPPORTED;
        } break;

        default:
            return STS_ERR_UNSUPPORTED;
        }

        OWN_SAFE_READ(&pHeader->biSizeImage, sizeof(pHeader->biSizeImage));
        OWN_SAFE_READ(&pHeader->biXPelsPerMeter, sizeof(pHeader->biXPelsPerMeter));
        OWN_SAFE_READ(&pHeader->biYPelsPerMeter, sizeof(pHeader->biYPelsPerMeter));
        OWN_SAFE_READ(&pHeader->biClrUsed, sizeof(pHeader->biClrUsed));
        OWN_SAFE_READ(&pHeader->biClrImportant, sizeof(pHeader->biClrImportant));
    }

    image.m_size.width = pHeader->biWidth;
    image.m_size.height = ABS(pHeader->biHeight);

    image.m_samples = pHeader->biBitCount >> 3;

    if (image.m_samples == 1)
        image.m_color = CF_GRAY;
    else if (image.m_samples == 3)
        image.m_color = CF_BGR;
    else if (image.m_samples == 4) {
        if (3L == pHeader->biCompression && pHeader->biBitCount == 32)
            image.m_color = CF_RGBA;
        else
            image.m_color = CF_BGRA;
    } else
        return STS_ERR_UNSUPPORTED;

    // Reinit image to properly set data
    image = Image(image.m_size, image.m_color, ST_8U);

    // Set position for reading data
    if (!fHeader.bfOffBits)
        return STS_ERR_FAILED;

    if (file.Seek(fHeader.bfOffBits, File::ORIGIN_BEGIN))
        return STS_ERR_FAILED;

    return STS_OK;
}

Status BmpReadData(File &file, Image &image)
{
    Status status;
    BMPInfoHeader header;

    status = BmpReadHeader(file, &header, image);
    if (status < 0)
        return status;

    // check for file size doesn't exceed allowed BMP_IMAGE_MAX_SIZE
    if ((BMP_IMAGE_MAX_SIZE < (size_t)(image.m_size.height * image.m_size.width * image.m_sampleSize * image.m_samples)) ||
        ((size_t)(image.m_size.height * image.m_size.width * image.m_sampleSize * image.m_samples) <= 0))
        return STS_ERR_UNSUPPORTED;

    image.Alloc();

    size_t iFileStep = alignValue<size_t>((size_t)(image.m_size.width * image.m_sampleSize * image.m_samples), 4);

    if (0 < header.biHeight) // Read bottom-up BMP
    {
        for (long long i = image.m_size.height; i > 0; i--) {
            OWN_SAFE_READ(image.ptr(i - 1), iFileStep);
        }
    } else // Read up-bottom BMP
    {
        for (long long i = 0; i < image.m_size.height; i++) {
            OWN_SAFE_READ(image.ptr(i), iFileStep);
        }
    }

    // Convert from ABGR to RGBA
    if (3L == header.biCompression && header.biBitCount == 32)
        cppiSwapChannesARGB<unsigned char>(image.ptr(), image.m_step, image.ptr(), image.m_step, image.m_size);

    return STS_OK;
}

Status BmpWriteData(Image &image, File &file)
{
    unsigned int iIHSize = 40;
    unsigned int iFHSize = 14;
    size_t iFileSize;
    size_t iImageSize;
    size_t iFileStep;

    RGBquad palette[256] = {0};
    BMPFileHeader fHeader;
    BMPInfoHeader header;

    iFileStep = alignValue<size_t>((size_t)(image.m_size.width * image.m_sampleSize * image.m_samples), 4);
    iImageSize = iFileStep * (size_t)image.m_size.height;
    iFileSize = iImageSize + iIHSize + iFHSize;

    fHeader.bfType = 0x4D42;
    fHeader.bfSize = (unsigned int)iFileSize;
    fHeader.bfReserved1 = 0;
    fHeader.bfReserved2 = 0;
    fHeader.bfOffBits = iIHSize + iFHSize;

    if (image.m_samples == 1)
        fHeader.bfOffBits += sizeof(palette);

    // write header
    OWN_SAFE_WRITE(&fHeader.bfType, sizeof(fHeader.bfType));
    OWN_SAFE_WRITE(&fHeader.bfSize, sizeof(fHeader.bfSize));
    OWN_SAFE_WRITE(&fHeader.bfReserved1, sizeof(fHeader.bfReserved1));
    OWN_SAFE_WRITE(&fHeader.bfReserved2, sizeof(fHeader.bfReserved2));
    OWN_SAFE_WRITE(&fHeader.bfOffBits, sizeof(fHeader.bfOffBits));

    header.biSize = iIHSize;
    header.biWidth = (unsigned int)image.m_size.width;
    header.biHeight = (unsigned int)image.m_size.height;
    header.biPlanes = 1;
    header.biBitCount = (unsigned short)(image.m_samples << 3);
    header.biCompression = 0L; // BI_RGB
    header.biSizeImage = (unsigned int)iImageSize;
    header.biXPelsPerMeter = 0;
    header.biYPelsPerMeter = 0;
    header.biClrUsed = ((image.m_samples == 1) ? 256 : 0);
    header.biClrImportant = ((image.m_samples == 1) ? 256 : 0);

    OWN_SAFE_WRITE(&header.biSize, sizeof(header.biSize));
    OWN_SAFE_WRITE(&header.biWidth, sizeof(header.biWidth));
    OWN_SAFE_WRITE(&header.biHeight, sizeof(header.biHeight));
    OWN_SAFE_WRITE(&header.biPlanes, sizeof(header.biPlanes));
    OWN_SAFE_WRITE(&header.biBitCount, sizeof(header.biBitCount));
    OWN_SAFE_WRITE(&header.biCompression, sizeof(header.biCompression));
    OWN_SAFE_WRITE(&header.biSizeImage, sizeof(header.biSizeImage));
    OWN_SAFE_WRITE(&header.biXPelsPerMeter, sizeof(header.biXPelsPerMeter));
    OWN_SAFE_WRITE(&header.biYPelsPerMeter, sizeof(header.biYPelsPerMeter));
    OWN_SAFE_WRITE(&header.biClrUsed, sizeof(header.biClrUsed));
    OWN_SAFE_WRITE(&header.biClrImportant, sizeof(header.biClrImportant));

    if (image.m_samples == 1) {
        for (int i = 0; i < 256; i++) {
            palette[i].rgbBlue = (unsigned char)i;
            palette[i].rgbGreen = (unsigned char)i;
            palette[i].rgbRed = (unsigned char)i;
            palette[i].rgbReserved = (unsigned char)0;
        }

        OWN_SAFE_WRITE(&palette[0], sizeof(palette));
    }

    // write data
    for (long long i = image.m_size.height; i > 0; i--) {
        OWN_SAFE_WRITE(image.ptr(i - 1), iFileStep);
    }

    return STS_OK;
}

#undef OWN_SAFE_READ
#undef OWN_SAFE_WRITE
