// Copyright (C) 2018 The Android Open Source Project
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include <benchmark/benchmark.h>

#include <optional>

#include "perfetto/base/flat_set.h"
#include "perfetto/ext/base/utils.h"
#include "perfetto/protozero/root_message.h"
#include "perfetto/protozero/scattered_stream_null_delegate.h"
#include "perfetto/protozero/scattered_stream_writer.h"
#include "src/traced/probes/ftrace/cpu_reader.h"
#include "src/traced/probes/ftrace/ftrace_config_muxer.h"
#include "src/traced/probes/ftrace/ftrace_print_filter.h"
#include "src/traced/probes/ftrace/proto_translation_table.h"
#include "src/traced/probes/ftrace/test/cpu_reader_support.h"
#include "src/tracing/core/null_trace_writer.h"

#include "protos/perfetto/trace/ftrace/ftrace_event_bundle.pbzero.h"
#include "protos/perfetto/trace/ftrace/ftrace_stats.pbzero.h"

namespace perfetto {
namespace {

using ::perfetto::protos::pbzero::FtraceEventBundle;
using ::protozero::ScatteredStreamWriter;
using ::protozero::ScatteredStreamWriterNullDelegate;

// A page full of "sched/sched_switch" events.
ExamplePage g_full_page_sched_switch{
    "synthetic",
    R"(
00000000: 31f2 7622 1a00 0000 b40f 0000 0000 0000  1.v"............
00000010: 1e00 0000 0000 0000 1000 0000 2f00 0103  ............/...
00000020: 140d 0000 4a69 7420 7468 7265 6164 2070  ....Jit thread p
00000030: 6f6f 6c00 140d 0000 8100 0000 0008 0000  ool.............
00000040: 0000 0000 4576 656e 7454 6872 6561 6400  ....EventThread.
00000050: 6572 0000 7002 0000 6100 0000 f057 0e00  er..p...a....W..
00000060: 2f00 0103 7002 0000 4576 656e 7454 6872  /...p...EventThr
00000070: 6561 6400 6572 0000 7002 0000 6100 0000  ead.er..p...a...
00000080: 0100 0000 0000 0000 4a69 7420 7468 7265  ........Jit thre
00000090: 6164 2070 6f6f 6c00 140d 0000 8100 0000  ad pool.........
000000a0: 50c2 0910 2f00 0103 140d 0000 4a69 7420  P.../.......Jit
000000b0: 7468 7265 6164 2070 6f6f 6c00 140d 0000  thread pool.....
000000c0: 8100 0000 0100 0000 0000 0000 7377 6170  ............swap
000000d0: 7065 722f 3000 0000 0000 0000 0000 0000  per/0...........
000000e0: 7800 0000 901a c80e 2f00 0103 0000 0000  x......./.......
000000f0: 7377 6170 7065 722f 3000 0000 0000 0000  swapper/0.......
00000100: 0000 0000 7800 0000 0000 0000 0000 0000  ....x...........
00000110: 4469 7370 5379 6e63 0069 6e67 6572 0000  DispSync.inger..
00000120: 6f02 0000 6100 0000 1064 1e00 2f00 0103  o...a....d../...
00000130: 6f02 0000 4469 7370 5379 6e63 0069 6e67  o...DispSync.ing
00000140: 6572 0000 6f02 0000 6100 0000 0100 0000  er..o...a.......
00000150: 0000 0000 7377 6170 7065 722f 3000 0000  ....swapper/0...
00000160: 0000 0000 0000 0000 7800 0000 9074 8600  ........x....t..
00000170: 2f00 0103 0000 0000 7377 6170 7065 722f  /.......swapper/
00000180: 3000 0000 0000 0000 0000 0000 7800 0000  0...........x...
00000190: 0000 0000 0000 0000 4576 656e 7454 6872  ........EventThr
000001a0: 6561 6400 6572 0000 7002 0000 6100 0000  ead.er..p...a...
000001b0: d071 0b00 2f00 0103 7002 0000 4576 656e  .q../...p...Even
000001c0: 7454 6872 6561 6400 6572 0000 7002 0000  tThread.er..p...
000001d0: 6100 0000 0100 0000 0000 0000 7377 6170  a...........swap
000001e0: 7065 722f 3000 0000 0000 0000 0000 0000  per/0...........
000001f0: 7800 0000 10cd 4504 2f00 0103 0000 0000  x.....E./.......
00000200: 7377 6170 7065 722f 3000 0000 0000 0000  swapper/0.......
00000210: 0000 0000 7800 0000 0000 0000 0000 0000  ....x...........
00000220: 7375 676f 763a 3000 0000 0000 0000 0000  sugov:0.........
00000230: 3802 0000 3100 0000 30d6 1300 2f00 0103  8...1...0.../...
00000240: 3802 0000 7375 676f 763a 3000 0000 0000  8...sugov:0.....
00000250: 0000 0000 3802 0000 3100 0000 0100 0000  ....8...1.......
00000260: 0000 0000 7377 6170 7065 722f 3000 0000  ....swapper/0...
00000270: 0000 0000 0000 0000 7800 0000 3049 a202  ........x...0I..
00000280: 2f00 0103 0000 0000 7377 6170 7065 722f  /.......swapper/
00000290: 3000 0000 0000 0000 0000 0000 7800 0000  0...........x...
000002a0: 0000 0000 0000 0000 4469 7370 5379 6e63  ........DispSync
000002b0: 0069 6e67 6572 0000 6f02 0000 6100 0000  .inger..o...a...
000002c0: d07a 1000 2f00 0103 6f02 0000 4469 7370  .z../...o...Disp
000002d0: 5379 6e63 0069 6e67 6572 0000 6f02 0000  Sync.inger..o...
000002e0: 6100 0000 0100 0000 0000 0000 7377 6170  a...........swap
000002f0: 7065 722f 3000 0000 0000 0000 0000 0000  per/0...........
00000300: 7800 0000 d085 1100 2f00 0103 0000 0000  x......./.......
00000310: 7377 6170 7065 722f 3000 0000 0000 0000  swapper/0.......
00000320: 0000 0000 7800 0000 0000 0000 0000 0000  ....x...........
00000330: 7375 7266 6163 6566 6c69 6e67 6572 0000  surfaceflinger..
00000340: 4b02 0000 6200 0000 907a f000 2f00 0103  K...b....z../...
00000350: 4b02 0000 7375 7266 6163 6566 6c69 6e67  K...surfacefling
00000360: 6572 0000 4b02 0000 6200 0000 0100 0000  er..K...b.......
00000370: 0000 0000 7377 6170 7065 722f 3000 0000  ....swapper/0...
00000380: 0000 0000 0000 0000 7800 0000 305a 6400  ........x...0Zd.
00000390: 2f00 0103 0000 0000 7377 6170 7065 722f  /.......swapper/
000003a0: 3000 0000 0000 0000 0000 0000 7800 0000  0...........x...
000003b0: 0000 0000 0000 0000 6d64 7373 5f66 6230  ........mdss_fb0
000003c0: 0000 0000 0000 0000 5714 0000 5300 0000  ........W...S...
000003d0: 10b1 9e03 2f00 0103 5714 0000 6d64 7373  ..../...W...mdss
000003e0: 5f66 6230 0000 0000 0000 0000 5714 0000  _fb0........W...
000003f0: 5300 0000 0200 0000 0000 0000 6b73 6f66  S...........ksof
00000400: 7469 7271 642f 3000 0000 0000 0300 0000  tirqd/0.........
00000410: 7800 0000 90bb 9900 2f00 0103 0300 0000  x......./.......
00000420: 6b73 6f66 7469 7271 642f 3000 0000 0000  ksoftirqd/0.....
00000430: 0300 0000 7800 0000 0100 0000 0000 0000  ....x...........
00000440: 7377 6170 7065 722f 3000 0000 0000 0000  swapper/0.......
00000450: 0000 0000 7800 0000 701e 5305 2f00 0103  ....x...p.S./...
00000460: 0000 0000 7377 6170 7065 722f 3000 0000  ....swapper/0...
00000470: 0000 0000 0000 0000 7800 0000 0000 0000  ........x.......
00000480: 0000 0000 6b77 6f72 6b65 722f 7531 363a  ....kworker/u16:
00000490: 3600 0000 6401 0000 7800 0000 90a1 2900  6...d...x.....).
000004a0: 2f00 0103 6401 0000 6b77 6f72 6b65 722f  /...d...kworker/
000004b0: 7531 363a 3600 0000 6401 0000 7800 0000  u16:6...d...x...
000004c0: 0200 0000 0000 0000 7377 6170 7065 722f  ........swapper/
000004d0: 3000 0000 0000 0000 0000 0000 7800 0000  0...........x...
000004e0: b0e5 4f04 2f00 0103 0000 0000 7377 6170  ..O./.......swap
000004f0: 7065 722f 3000 0000 0000 0000 0000 0000  per/0...........
00000500: 7800 0000 0000 0000 0000 0000 4269 6e64  x...........Bind
00000510: 6572 3a32 3136 385f 3135 0000 e614 0000  er:2168_15......
00000520: 7800 0000 b0bd 7c00 2f00 0103 e614 0000  x.....|./.......
00000530: 4269 6e64 6572 3a32 3136 385f 3135 0000  Binder:2168_15..
00000540: e614 0000 7800 0000 0100 0000 0000 0000  ....x...........
00000550: 7377 6170 7065 722f 3000 0000 0000 0000  swapper/0.......
00000560: 0000 0000 7800 0000 d0bd 7e01 2f00 0103  ....x.....~./...
00000570: 0000 0000 7377 6170 7065 722f 3000 0000  ....swapper/0...
00000580: 0000 0000 0000 0000 7800 0000 0000 0000  ........x.......
00000590: 0000 0000 6b77 6f72 6b65 722f 7531 363a  ....kworker/u16:
000005a0: 3900 0000 e204 0000 7800 0000 7016 0800  9.......x...p...
000005b0: 2f00 0103 e204 0000 6b77 6f72 6b65 722f  /.......kworker/
000005c0: 7531 363a 3900 0000 e204 0000 7800 0000  u16:9.......x...
000005d0: 0100 0000 0000 0000 7377 6170 7065 722f  ........swapper/
000005e0: 3000 0000 0000 0000 0000 0000 7800 0000  0...........x...
000005f0: 1004 5200 2f00 0103 0000 0000 7377 6170  ..R./.......swap
00000600: 7065 722f 3000 0000 0000 0000 0000 0000  per/0...........
00000610: 7800 0000 0000 0000 0000 0000 6b77 6f72  x...........kwor
00000620: 6b65 722f 7531 363a 3900 0000 e204 0000  ker/u16:9.......
00000630: 7800 0000 d0db 0700 2f00 0103 e204 0000  x......./.......
00000640: 6b77 6f72 6b65 722f 7531 363a 3900 0000  kworker/u16:9...
00000650: e204 0000 7800 0000 0100 0000 0000 0000  ....x...........
00000660: 7377 6170 7065 722f 3000 0000 0000 0000  swapper/0.......
00000670: 0000 0000 7800 0000 b0a2 8c00 2f00 0103  ....x......./...
00000680: 0000 0000 7377 6170 7065 722f 3000 0000  ....swapper/0...
00000690: 0000 0000 0000 0000 7800 0000 0000 0000  ........x.......
000006a0: 0000 0000 6b77 6f72 6b65 722f 7531 363a  ....kworker/u16:
000006b0: 3900 0000 e204 0000 7800 0000 d02b 0400  9.......x....+..
000006c0: 2f00 0103 e204 0000 6b77 6f72 6b65 722f  /.......kworker/
000006d0: 7531 363a 3900 0000 e204 0000 7800 0000  u16:9.......x...
000006e0: 0100 0000 0000 0000 7377 6170 7065 722f  ........swapper/
000006f0: 3000 0000 0000 0000 0000 0000 7800 0000  0...........x...
00000700: d064 ef05 2f00 0103 0000 0000 7377 6170  .d../.......swap
00000710: 7065 722f 3000 0000 0000 0000 0000 0000  per/0...........
00000720: 7800 0000 0000 0000 0000 0000 4469 7370  x...........Disp
00000730: 5379 6e63 0069 6e67 6572 0000 6f02 0000  Sync.inger..o...
00000740: 6100 0000 f07d 1b00 2f00 0103 6f02 0000  a....}../...o...
00000750: 4469 7370 5379 6e63 0069 6e67 6572 0000  DispSync.inger..
00000760: 6f02 0000 6100 0000 0100 0000 0000 0000  o...a...........
00000770: 6b73 6f66 7469 7271 642f 3000 0000 0000  ksoftirqd/0.....
00000780: 0300 0000 7800 0000 304c 2000 2f00 0103  ....x...0L ./...
00000790: 0300 0000 6b73 6f66 7469 7271 642f 3000  ....ksoftirqd/0.
000007a0: 0000 0000 0300 0000 7800 0000 0100 0000  ........x.......
000007b0: 0000 0000 6465 7832 6f61 7400 3935 5f33  ....dex2oat.95_3
000007c0: 0000 0000 341f 0000 8200 0000 700b 0700  ....4.......p...
000007d0: 2f00 0103 341f 0000 6465 7832 6f61 7400  /...4...dex2oat.
000007e0: 3935 5f33 0000 0000 341f 0000 8200 0000  95_3....4.......
000007f0: 0000 0000 0000 0000 7375 676f 763a 3000  ........sugov:0.
00000800: 0000 0000 0000 0000 3802 0000 3100 0000  ........8...1...
00000810: 50b0 0600 2f00 0103 3802 0000 7375 676f  P.../...8...sugo
00000820: 763a 3000 0000 0000 0000 0000 3802 0000  v:0.........8...
00000830: 3100 0000 0008 0000 0000 0000 6d69 6772  1...........migr
00000840: 6174 696f 6e2f 3000 0000 0000 0d00 0000  ation/0.........
00000850: 0000 0000 d09c 0600 2f00 0103 0d00 0000  ......../.......
00000860: 6d69 6772 6174 696f 6e2f 3000 0000 0000  migration/0.....
00000870: 0d00 0000 0000 0000 0100 0000 0000 0000  ................
00000880: 7375 676f 763a 3000 0000 0000 0000 0000  sugov:0.........
00000890: 3802 0000 3100 0000 7061 1900 2f00 0103  8...1...pa../...
000008a0: 3802 0000 7375 676f 763a 3000 0000 0000  8...sugov:0.....
000008b0: 0000 0000 3802 0000 3100 0000 0100 0000  ....8...1.......
000008c0: 0000 0000 6465 7832 6f61 7400 3935 5f33  ....dex2oat.95_3
000008d0: 0000 0000 341f 0000 8200 0000 f03c 5600  ....4........<V.
000008e0: 2f00 0103 341f 0000 6465 7832 6f61 7400  /...4...dex2oat.
000008f0: 3935 5f33 0000 0000 341f 0000 8200 0000  95_3....4.......
00000900: 0200 0000 0000 0000 7377 6170 7065 722f  ........swapper/
00000910: 3000 0000 0000 0000 0000 0000 7800 0000  0...........x...
00000920: 5013 c400 2f00 0103 0000 0000 7377 6170  P.../.......swap
00000930: 7065 722f 3000 0000 0000 0000 0000 0000  per/0...........
00000940: 7800 0000 0000 0000 0000 0000 616e 6472  x...........andr
00000950: 6f69 642e 6861 7264 7761 7200 d20a 0000  oid.hardwar.....
00000960: 7800 0000 30c9 1300 2f00 0103 d20a 0000  x...0.../.......
00000970: 616e 6472 6f69 642e 6861 7264 7761 7200  android.hardwar.
00000980: d20a 0000 7800 0000 0100 0000 0000 0000  ....x...........
00000990: 7377 6170 7065 722f 3000 0000 0000 0000  swapper/0.......
000009a0: 0000 0000 7800 0000 7097 c000 2f00 0103  ....x...p.../...
000009b0: 0000 0000 7377 6170 7065 722f 3000 0000  ....swapper/0...
000009c0: 0000 0000 0000 0000 7800 0000 0000 0000  ........x.......
000009d0: 0000 0000 616e 6472 6f69 642e 6861 7264  ....android.hard
000009e0: 7761 7200 d20a 0000 7800 0000 305c 0c00  war.....x...0\..
000009f0: 2f00 0103 d20a 0000 616e 6472 6f69 642e  /.......android.
00000a00: 6861 7264 7761 7200 d20a 0000 7800 0000  hardwar.....x...
00000a10: 0100 0000 0000 0000 7377 6170 7065 722f  ........swapper/
00000a20: 3000 0000 0000 0000 0000 0000 7800 0000  0...........x...
00000a30: d0aa 1401 2f00 0103 0000 0000 7377 6170  ..../.......swap
00000a40: 7065 722f 3000 0000 0000 0000 0000 0000  per/0...........
00000a50: 7800 0000 0000 0000 0000 0000 616e 6472  x...........andr
00000a60: 6f69 642e 6861 7264 7761 7200 d20a 0000  oid.hardwar.....
00000a70: 7800 0000 903b 0c00 2f00 0103 d20a 0000  x....;../.......
00000a80: 616e 6472 6f69 642e 6861 7264 7761 7200  android.hardwar.
00000a90: d20a 0000 7800 0000 0100 0000 0000 0000  ....x...........
00000aa0: 7377 6170 7065 722f 3000 0000 0000 0000  swapper/0.......
00000ab0: 0000 0000 7800 0000 f024 5401 2f00 0103  ....x....$T./...
00000ac0: 0000 0000 7377 6170 7065 722f 3000 0000  ....swapper/0...
00000ad0: 0000 0000 0000 0000 7800 0000 0000 0000  ........x.......
00000ae0: 0000 0000 616e 6472 6f69 642e 6861 7264  ....android.hard
00000af0: 7761 7200 d20a 0000 7800 0000 f0f3 0b00  war.....x.......
00000b00: 2f00 0103 d20a 0000 616e 6472 6f69 642e  /.......android.
00000b10: 6861 7264 7761 7200 d20a 0000 7800 0000  hardwar.....x...
00000b20: 0100 0000 0000 0000 7377 6170 7065 722f  ........swapper/
00000b30: 3000 0000 0000 0000 0000 0000 7800 0000  0...........x...
00000b40: d0b5 bf02 2f00 0103 0000 0000 7377 6170  ..../.......swap
00000b50: 7065 722f 3000 0000 0000 0000 0000 0000  per/0...........
00000b60: 7800 0000 0000 0000 0000 0000 4469 7370  x...........Disp
00000b70: 5379 6e63 0069 6e67 6572 0000 6f02 0000  Sync.inger..o...
00000b80: 6100 0000 90cd 1400 2f00 0103 6f02 0000  a......./...o...
00000b90: 4469 7370 5379 6e63 0069 6e67 6572 0000  DispSync.inger..
00000ba0: 6f02 0000 6100 0000 0100 0000 0000 0000  o...a...........
00000bb0: 7377 6170 7065 722f 3000 0000 0000 0000  swapper/0.......
00000bc0: 0000 0000 7800 0000 50a6 1100 2f00 0103  ....x...P.../...
00000bd0: 0000 0000 7377 6170 7065 722f 3000 0000  ....swapper/0...
00000be0: 0000 0000 0000 0000 7800 0000 0000 0000  ........x.......
00000bf0: 0000 0000 7375 7266 6163 6566 6c69 6e67  ....surfacefling
00000c00: 6572 0000 4b02 0000 6200 0000 b04c 4200  er..K...b....LB.
00000c10: 2f00 0103 4b02 0000 7375 7266 6163 6566  /...K...surfacef
00000c20: 6c69 6e67 6572 0000 4b02 0000 6200 0000  linger..K...b...
00000c30: 0100 0000 0000 0000 7377 6170 7065 722f  ........swapper/
00000c40: 3000 0000 0000 0000 0000 0000 7800 0000  0...........x...
00000c50: b025 060a 2f00 0103 0000 0000 7377 6170  .%../.......swap
00000c60: 7065 722f 3000 0000 0000 0000 0000 0000  per/0...........
00000c70: 7800 0000 0000 0000 0000 0000 6b77 6f72  x...........kwor
00000c80: 6b65 722f 7531 363a 3600 0000 6401 0000  ker/u16:6...d...
00000c90: 7800 0000 d0b6 0600 2f00 0103 6401 0000  x......./...d...
00000ca0: 6b77 6f72 6b65 722f 7531 363a 3600 0000  kworker/u16:6...
00000cb0: 6401 0000 7800 0000 0100 0000 0000 0000  d...x...........
00000cc0: 7377 6170 7065 722f 3000 0000 0000 0000  swapper/0.......
00000cd0: 0000 0000 7800 0000 f0a0 5800 2f00 0103  ....x.....X./...
00000ce0: 0000 0000 7377 6170 7065 722f 3000 0000  ....swapper/0...
00000cf0: 0000 0000 0000 0000 7800 0000 0000 0000  ........x.......
00000d00: 0000 0000 6b77 6f72 6b65 722f 7531 363a  ....kworker/u16:
00000d10: 3600 0000 6401 0000 7800 0000 f07a 1300  6...d...x....z..
00000d20: 2f00 0103 6401 0000 6b77 6f72 6b65 722f  /...d...kworker/
00000d30: 7531 363a 3600 0000 6401 0000 7800 0000  u16:6...d...x...
00000d40: 0100 0000 0000 0000 7377 6170 7065 722f  ........swapper/
00000d50: 3000 0000 0000 0000 0000 0000 7800 0000  0...........x...
00000d60: b080 b101 2f00 0103 0000 0000 7377 6170  ..../.......swap
00000d70: 7065 722f 3000 0000 0000 0000 0000 0000  per/0...........
00000d80: 7800 0000 0000 0000 0000 0000 6b77 6f72  x...........kwor
00000d90: 6b65 722f 7531 363a 3600 0000 6401 0000  ker/u16:6...d...
00000da0: 7800 0000 103c 1200 2f00 0103 6401 0000  x....<../...d...
00000db0: 6b77 6f72 6b65 722f 7531 363a 3600 0000  kworker/u16:6...
00000dc0: 6401 0000 7800 0000 0100 0000 0000 0000  d...x...........
00000dd0: 7377 6170 7065 722f 3000 0000 0000 0000  swapper/0.......
00000de0: 0000 0000 7800 0000 50ea 3800 2f00 0103  ....x...P.8./...
00000df0: 0000 0000 7377 6170 7065 722f 3000 0000  ....swapper/0...
00000e00: 0000 0000 0000 0000 7800 0000 0000 0000  ........x.......
00000e10: 0000 0000 6b77 6f72 6b65 722f 7531 363a  ....kworker/u16:
00000e20: 3600 0000 6401 0000 7800 0000 5032 0400  6...d...x...P2..
00000e30: 2f00 0103 6401 0000 6b77 6f72 6b65 722f  /...d...kworker/
00000e40: 7531 363a 3600 0000 6401 0000 7800 0000  u16:6...d...x...
00000e50: 0100 0000 0000 0000 7377 6170 7065 722f  ........swapper/
00000e60: 3000 0000 0000 0000 0000 0000 7800 0000  0...........x...
00000e70: 70f5 9000 2f00 0103 0000 0000 7377 6170  p.../.......swap
00000e80: 7065 722f 3000 0000 0000 0000 0000 0000  per/0...........
00000e90: 7800 0000 0000 0000 0000 0000 6b77 6f72  x...........kwor
00000ea0: 6b65 722f 7531 363a 3600 0000 6401 0000  ker/u16:6...d...
00000eb0: 7800 0000 10d7 0300 2f00 0103 6401 0000  x......./...d...
00000ec0: 6b77 6f72 6b65 722f 7531 363a 3600 0000  kworker/u16:6...
00000ed0: 6401 0000 7800 0000 0100 0000 0000 0000  d...x...........
00000ee0: 7377 6170 7065 722f 3000 0000 0000 0000  swapper/0.......
00000ef0: 0000 0000 7800 0000 907c 0900 2f00 0103  ....x....|../...
00000f00: 0000 0000 7377 6170 7065 722f 3000 0000  ....swapper/0...
00000f10: 0000 0000 0000 0000 7800 0000 0000 0000  ........x.......
00000f20: 0000 0000 6b77 6f72 6b65 722f 7531 363a  ....kworker/u16:
00000f30: 3600 0000 6401 0000 7800 0000 7082 0300  6...d...x...p...
00000f40: 2f00 0103 6401 0000 6b77 6f72 6b65 722f  /...d...kworker/
00000f50: 7531 363a 3600 0000 6401 0000 7800 0000  u16:6...d...x...
00000f60: 0100 0000 0000 0000 7377 6170 7065 722f  ........swapper/
00000f70: 3000 0000 0000 0000 0000 0000 7800 0000  0...........x...
00000f80: f0ec 2100 2f00 0103 0000 0000 7377 6170  ..!./.......swap
00000f90: 7065 722f 3000 0000 0000 0000 0000 0000  per/0...........
00000fa0: 7800 0000 0000 0000 0000 0000 6b77 6f72  x...........kwor
00000fb0: 6b65 722f 7531 363a 3600 0000 6401 0000  ker/u16:6...d...
00000fc0: 7800 0000 0000 0000 0000 0000 0000 0000  x...............
00000fd0: 0000 0000 0000 0000 0000 0000 0000 0000  ................
00000fe0: 0000 0000 0000 0000 0000 0000 0000 0000  ................
00000ff0: 0000 0000 0000 0000 0000 0000 0000 0000  ................
    )",
};

// A page full of "ftrace/print" events.
ExamplePage g_full_page_print{
    "synthetic",
    R"(
00000000: df34 b6e2 e2a9 0000 e00f 0000 0000 0000  .4..............
00000010: 0700 0000 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000020: ffff ffff 3333 3333 0a00 0000 f348 0700  ....3333.....H..
00000030: 47d0 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
00000040: ffff ffff 3434 3434 0a00 722f 7531 3435  ....4444..r/u145
00000050: 07ab 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000060: ffff ffff 3535 3535 0a00 0700 6b77 6f72  ....5555....kwor
00000070: c7b0 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000080: ffff ffff 3636 3636 0a00 0100 4001 0103  ....6666....@...
00000090: a7b8 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000000a0: ffff ffff 3737 3737 0a00 0000 0100 0000  ....7777........
000000b0: 47ac 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
000000c0: ffff ffff 3838 3838 0a00 0000 f348 0700  ....8888.....H..
000000d0: 47d9 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
000000e0: ffff ffff 3939 3939 0a00 722f 7531 3435  ....9999..r/u145
000000f0: c7b9 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000100: ffff ffff 4141 4141 0a00 0700 6b77 6f72  ....AAAA....kwor
00000110: c7b6 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000120: ffff ffff 4242 4242 0a00 0100 4001 0103  ....BBBB....@...
00000130: c7a6 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000140: ffff ffff 4343 4343 0a00 0000 0100 0000  ....CCCC........
00000150: c7ab 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000160: ffff ffff 4444 4444 0a00 0000 f348 0700  ....DDDD.....H..
00000170: 27b3 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
00000180: ffff ffff 4545 4545 0a00 722f 7531 3435  ....EEEE..r/u145
00000190: 27b0 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
000001a0: ffff ffff 4646 4646 0a00 0700 6b77 6f72  ....FFFF....kwor
000001b0: c778 0c00 0500 0000 4f56 0700 84fc 9a9f  .x......OV......
000001c0: ffff ffff 3030 3030 0a00 0100 4001 0103  ....0000....@...
000001d0: 27b8 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
000001e0: ffff ffff 3131 3131 0a00 0000 0100 0000  ....1111........
000001f0: 27af 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
00000200: ffff ffff 3232 3232 0a00 0000 f348 0700  ....2222.....H..
00000210: 27a5 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
00000220: ffff ffff 3333 3333 0a00 722f 7531 3435  ....3333..r/u145
00000230: 07b7 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000240: ffff ffff 3434 3434 0a00 0700 6b77 6f72  ....4444....kwor
00000250: 47ac 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
00000260: ffff ffff 3535 3535 0a00 0100 4001 0103  ....5555....@...
00000270: 27ad 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
00000280: ffff ffff 3636 3636 0a00 0000 0100 0000  ....6666........
00000290: e7b6 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000002a0: ffff ffff 3737 3737 0a00 0000 f348 0700  ....7777.....H..
000002b0: 47b1 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
000002c0: ffff ffff 3838 3838 0a00 722f 7531 3435  ....8888..r/u145
000002d0: a7a7 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000002e0: ffff ffff 3939 3939 0a00 0700 6b77 6f72  ....9999....kwor
000002f0: 87a9 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000300: ffff ffff 4141 4141 0a00 0100 4001 0103  ....AAAA....@...
00000310: 27cb 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
00000320: ffff ffff 4242 4242 0a00 0000 0100 0000  ....BBBB........
00000330: 47c0 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
00000340: ffff ffff 4343 4343 0a00 0000 f348 0700  ....CCCC.....H..
00000350: 67ae 0900 0500 0000 4f56 0700 84fc 9a9f  g.......OV......
00000360: ffff ffff 4444 4444 0a00 722f 7531 3435  ....DDDD..r/u145
00000370: 47b0 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
00000380: ffff ffff 4545 4545 0a00 0700 6b77 6f72  ....EEEE....kwor
00000390: 87a6 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000003a0: ffff ffff 4646 4646 0a00 0100 4001 0103  ....FFFF....@...
000003b0: 07a2 0c00 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000003c0: ffff ffff 3030 3030 0a00 0000 0100 0000  ....0000........
000003d0: e71f 0a00 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000003e0: ffff ffff 3131 3131 0a00 0000 f348 0700  ....1111.....H..
000003f0: 67c6 0900 0500 0000 4f56 0700 84fc 9a9f  g.......OV......
00000400: ffff ffff 3232 3232 0a00 722f 7531 3435  ....2222..r/u145
00000410: 67b2 0900 0500 0000 4f56 0700 84fc 9a9f  g.......OV......
00000420: ffff ffff 3333 3333 0a00 0700 6b77 6f72  ....3333....kwor
00000430: 27af 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
00000440: ffff ffff 3434 3434 0a00 0100 4001 0103  ....4444....@...
00000450: 07d8 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000460: ffff ffff 3535 3535 0a00 0000 0100 0000  ....5555........
00000470: 87ba 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000480: ffff ffff 3636 3636 0a00 0000 f348 0700  ....6666.....H..
00000490: 27c3 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
000004a0: ffff ffff 3737 3737 0a00 722f 7531 3435  ....7777..r/u145
000004b0: 27bb 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
000004c0: ffff ffff 3838 3838 0a00 0700 6b77 6f72  ....8888....kwor
000004d0: c7a6 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000004e0: ffff ffff 3939 3939 0a00 0100 4001 0103  ....9999....@...
000004f0: e7ac 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000500: ffff ffff 4141 4141 0a00 0000 0100 0000  ....AAAA........
00000510: e7a5 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000520: ffff ffff 4242 4242 0a00 0000 f348 0700  ....BBBB.....H..
00000530: c7ab 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000540: ffff ffff 4343 4343 0a00 722f 7531 3435  ....CCCC..r/u145
00000550: 07aa 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000560: ffff ffff 4444 4444 0a00 0700 6b77 6f72  ....DDDD....kwor
00000570: 27b1 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
00000580: ffff ffff 4545 4545 0a00 0100 4001 0103  ....EEEE....@...
00000590: e7ae 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000005a0: ffff ffff 4646 4646 0a00 0000 0100 0000  ....FFFF........
000005b0: 876e 0c00 0500 0000 4f56 0700 84fc 9a9f  .n......OV......
000005c0: ffff ffff 3030 3030 0a00 0000 f348 0700  ....0000.....H..
000005d0: 87b0 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000005e0: ffff ffff 3131 3131 0a00 722f 7531 3435  ....1111..r/u145
000005f0: a7de 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000600: ffff ffff 3232 3232 0a00 0700 6b77 6f72  ....2222....kwor
00000610: c7bf 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000620: ffff ffff 3333 3333 0a00 0100 4001 0103  ....3333....@...
00000630: e7ad 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000640: ffff ffff 3434 3434 0a00 0000 2e00 0000  ....4444........
00000650: 67c2 0900 0500 0000 4f56 0700 84fc 9a9f  g.......OV......
00000660: ffff ffff 3535 3535 0a00 0000 f348 0700  ....5555.....H..
00000670: 87ec 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000680: ffff ffff 3636 3636 0a00 722f 7531 3435  ....6666..r/u145
00000690: e7b9 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000006a0: ffff ffff 3737 3737 0a00 0700 6b77 6f72  ....7777....kwor
000006b0: e7aa 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000006c0: ffff ffff 3838 3838 0a00 0100 4001 0103  ....8888....@...
000006d0: 07a9 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000006e0: ffff ffff 3939 3939 0a00 0000 0100 0000  ....9999........
000006f0: 87a6 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000700: ffff ffff 4141 4141 0a00 0000 f348 0700  ....AAAA.....H..
00000710: c7a6 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000720: ffff ffff 4242 4242 0a00 722f 7531 3435  ....BBBB..r/u145
00000730: 67d0 0900 0500 0000 4f56 0700 84fc 9a9f  g.......OV......
00000740: ffff ffff 4343 4343 0a00 0700 6b77 6f72  ....CCCC....kwor
00000750: 27b6 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
00000760: ffff ffff 4444 4444 0a00 0100 4001 0103  ....DDDD....@...
00000770: e7ba 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000780: ffff ffff 4545 4545 0a00 0000 0100 0000  ....EEEE........
00000790: 07ce 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000007a0: ffff ffff 4646 4646 0a00 0000 f348 0700  ....FFFF.....H..
000007b0: c768 0c00 0500 0000 4f56 0700 84fc 9a9f  .h......OV......
000007c0: ffff ffff 3030 3030 0a00 722f 7531 3435  ....0000..r/u145
000007d0: 67de 0900 0500 0000 4f56 0700 84fc 9a9f  g.......OV......
000007e0: ffff ffff 3131 3131 0a00 0700 6b77 6f72  ....1111....kwor
000007f0: 27b9 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
00000800: ffff ffff 3232 3232 0a00 0100 4001 0103  ....2222....@...
00000810: c7b2 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000820: ffff ffff 3333 3333 0a00 0000 0100 0000  ....3333........
00000830: 67b1 0900 0500 0000 4f56 0700 84fc 9a9f  g.......OV......
00000840: ffff ffff 3434 3434 0a00 0000 f348 0700  ....4444.....H..
00000850: c7b2 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000860: ffff ffff 3535 3535 0a00 722f 7531 3435  ....5555..r/u145
00000870: c7c1 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000880: ffff ffff 3636 3636 0a00 0700 6b77 6f72  ....6666....kwor
00000890: 87ac 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000008a0: ffff ffff 3737 3737 0a00 0100 4001 0103  ....7777....@...
000008b0: c7b2 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000008c0: ffff ffff 3838 3838 0a00 0000 0100 0000  ....8888........
000008d0: 879e 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000008e0: ffff ffff 3939 3939 0a00 0000 f348 0700  ....9999.....H..
000008f0: 67ae 0900 0500 0000 4f56 0700 84fc 9a9f  g.......OV......
00000900: ffff ffff 4141 4141 0a00 722f 7531 3435  ....AAAA..r/u145
00000910: 07a6 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000920: ffff ffff 4242 4242 0a00 0700 6b77 6f72  ....BBBB....kwor
00000930: a7a2 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000940: ffff ffff 4343 4343 0a00 0100 4001 0103  ....CCCC....@...
00000950: 67a4 0900 0500 0000 4f56 0700 84fc 9a9f  g.......OV......
00000960: ffff ffff 4444 4444 0a00 0000 0100 0000  ....DDDD........
00000970: 87a5 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000980: ffff ffff 4545 4545 0a00 0000 5417 0700  ....EEEE....T...
00000990: 47b1 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
000009a0: ffff ffff 4646 4646 0a00 722f 7531 3435  ....FFFF..r/u145
000009b0: e75e 0c00 0500 0000 4f56 0700 84fc 9a9f  .^......OV......
000009c0: ffff ffff 3030 3030 0a00 0700 6b77 6f72  ....0000....kwor
000009d0: a7bc 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
000009e0: ffff ffff 3131 3131 0a00 0100 4001 0103  ....1111....@...
000009f0: c7a8 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000a00: ffff ffff 3232 3232 0a00 0000 2e00 0000  ....2222........
00000a10: c7af 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000a20: ffff ffff 3333 3333 0a00 0000 5417 0700  ....3333....T...
00000a30: 67aa 0900 0500 0000 4f56 0700 84fc 9a9f  g.......OV......
00000a40: ffff ffff 3434 3434 0a00 722f 7531 3435  ....4444..r/u145
00000a50: 07bd 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000a60: ffff ffff 3535 3535 0a00 0700 6b77 6f72  ....5555....kwor
00000a70: 67b2 0900 0500 0000 4f56 0700 84fc 9a9f  g.......OV......
00000a80: ffff ffff 3636 3636 0a00 0100 4001 0103  ....6666....@...
00000a90: 27b2 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
00000aa0: ffff ffff 3737 3737 0a00 0000 2e00 0000  ....7777........
00000ab0: a731 0a00 0500 0000 4f56 0700 84fc 9a9f  .1......OV......
00000ac0: ffff ffff 3838 3838 0a00 0000 5417 0700  ....8888....T...
00000ad0: e7bd 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000ae0: ffff ffff 3939 3939 0a00 722f 7531 3435  ....9999..r/u145
00000af0: c7ad 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000b00: ffff ffff 4141 4141 0a00 0700 6b77 6f72  ....AAAA....kwor
00000b10: 47af 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
00000b20: ffff ffff 4242 4242 0a00 0100 4001 0103  ....BBBB....@...
00000b30: c7d2 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000b40: ffff ffff 4343 4343 0a00 0000 2e00 0000  ....CCCC........
00000b50: 67a6 0900 0500 0000 4f56 0700 84fc 9a9f  g.......OV......
00000b60: ffff ffff 4444 4444 0a00 0000 5417 0700  ....DDDD....T...
00000b70: 47aa 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
00000b80: ffff ffff 4545 4545 0a00 722f 7531 3435  ....EEEE..r/u145
00000b90: 07b6 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000ba0: ffff ffff 4646 4646 0a00 0700 6b77 6f72  ....FFFF....kwor
00000bb0: 876b 0c00 0500 0000 4f56 0700 84fc 9a9f  .k......OV......
00000bc0: ffff ffff 3030 3030 0a00 0100 4001 0103  ....0000....@...
00000bd0: 47bc 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
00000be0: ffff ffff 3131 3131 0a00 0000 2e00 0000  ....1111........
00000bf0: a7b1 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000c00: ffff ffff 3232 3232 0a00 0000 5417 0700  ....2222....T...
00000c10: a7bb 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000c20: ffff ffff 3333 3333 0a00 722f 7531 3435  ....3333..r/u145
00000c30: 47b4 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
00000c40: ffff ffff 3434 3434 0a00 0700 6b77 6f72  ....4444....kwor
00000c50: 0710 0a00 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000c60: ffff ffff 3535 3535 0a00 0100 4001 0103  ....5555....@...
00000c70: 27cb 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
00000c80: ffff ffff 3636 3636 0a00 0000 2e00 0000  ....6666........
00000c90: c7a8 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000ca0: ffff ffff 3737 3737 0a00 0000 5417 0700  ....7777....T...
00000cb0: 47c6 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
00000cc0: ffff ffff 3838 3838 0a00 722f 7531 3435  ....8888..r/u145
00000cd0: 07c1 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000ce0: ffff ffff 3939 3939 0a00 0700 6b77 6f72  ....9999....kwor
00000cf0: 47b3 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
00000d00: ffff ffff 4141 4141 0a00 0300 4001 0103  ....AAAA....@...
00000d10: a7ad 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000d20: ffff ffff 4242 4242 0a00 0000 2e00 0000  ....BBBB........
00000d30: 87b3 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000d40: ffff ffff 4343 4343 0a00 0000 5417 0700  ....CCCC....T...
00000d50: 07b8 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000d60: ffff ffff 4444 4444 0a00 722f 7531 3435  ....DDDD..r/u145
00000d70: 07b6 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000d80: ffff ffff 4545 4545 0a00 0700 6b77 6f72  ....EEEE....kwor
00000d90: 27b4 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
00000da0: ffff ffff 4646 4646 0a00 0100 4001 0103  ....FFFF....@...
00000db0: 2761 0c00 0500 0000 4f56 0700 84fc 9a9f  'a......OV......
00000dc0: ffff ffff 3030 3030 0a00 0000 2e00 0000  ....0000........
00000dd0: 87b6 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000de0: ffff ffff 3131 3131 0a00 0000 5417 0700  ....1111....T...
00000df0: 87ae 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000e00: ffff ffff 3232 3232 0a00 722f 7531 3435  ....2222..r/u145
00000e10: c7ac 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000e20: ffff ffff 3333 3333 0a00 0700 6b77 6f72  ....3333....kwor
00000e30: 27c0 0900 0500 0000 4f56 0700 84fc 9a9f  '.......OV......
00000e40: ffff ffff 3434 3434 0a00 0100 4001 0103  ....4444....@...
00000e50: 07b2 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000e60: ffff ffff 3535 3535 0a00 0000 2e00 0000  ....5555........
00000e70: 67ae 0900 0500 0000 4f56 0700 84fc 9a9f  g.......OV......
00000e80: ffff ffff 3636 3636 0a00 0000 5417 0700  ....6666....T...
00000e90: a7af 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000ea0: ffff ffff 3737 3737 0a00 722f 7531 3435  ....7777..r/u145
00000eb0: e7aa 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000ec0: ffff ffff 3838 3838 0a00 0700 6b77 6f72  ....8888....kwor
00000ed0: 07ae 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000ee0: ffff ffff 3939 3939 0a00 0100 4001 0103  ....9999....@...
00000ef0: 87af 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000f00: ffff ffff 4141 4141 0a00 0000 2e00 0000  ....AAAA........
00000f10: 07ae 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000f20: ffff ffff 4242 4242 0a00 0000 5417 0700  ....BBBB....T...
00000f30: 47a0 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
00000f40: ffff ffff 4343 4343 0a00 722f 7531 3435  ....CCCC..r/u145
00000f50: e7d5 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000f60: ffff ffff 4444 4444 0a00 0700 6b77 6f72  ....DDDD....kwor
00000f70: 87bc 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000f80: ffff ffff 4545 4545 0a00 0100 4001 0103  ....EEEE....@...
00000f90: e7c3 0900 0500 0000 4f56 0700 84fc 9a9f  ........OV......
00000fa0: ffff ffff 4646 4646 0a00 0000 2e00 0000  ....FFFF........
00000fb0: 475a 0c00 0500 0000 4f56 0700 84fc 9a9f  GZ......OV......
00000fc0: ffff ffff 3030 3030 0a00 0000 5417 0700  ....0000....T...
00000fd0: 47bc 0900 0500 0000 4f56 0700 84fc 9a9f  G.......OV......
00000fe0: ffff ffff 3131 3131 0a00 722f 7531 3435  ....1111..r/u145
00000ff0: 0000 0000 0000 0000 0000 0000 0000 0000  ................
    )",
};

// A page full of "ftrace/print" atrace "C|pid|0000" events
ExamplePage g_full_page_atrace_print{
    "synthetic",
    R"(
00000000: 5c07 2181 3e14 0800 e40f 00c0 ffff ffff  \.!.>...........
00000010: 0800 0000 0500 0000 39ec 1700 042e 7b99  ........9.....{.
00000020: ffff ffff 437c 3131 317c 3333 3333 0a00  ....C|111|3333..
00000030: 0000 0000 8836 0c00 0500 0000 39ec 1700  .....6......9...
00000040: 042e 7b99 ffff ffff 437c 3131 317c 3434  ..{.....C|111|44
00000050: 3434 0a00 0000 0000 c84e 0c00 0500 0000  44.......N......
00000060: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000070: 317c 3535 3535 0a00 0000 0000 4847 0c00  1|5555......HG..
00000080: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000090: 437c 3131 317c 3636 3636 0a00 0000 0000  C|111|6666......
000000a0: 4835 0c00 0500 0000 39ec 1700 042e 7b99  H5......9.....{.
000000b0: ffff ffff 437c 3131 317c 3737 3737 0a00  ....C|111|7777..
000000c0: 0000 0000 8830 0c00 0500 0000 39ec 1700  .....0......9...
000000d0: 042e 7b99 ffff ffff 437c 3131 317c 3838  ..{.....C|111|88
000000e0: 3838 0a00 0000 0000 683d 0c00 0500 0000  88......h=......
000000f0: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000100: 317c 3939 3939 0a00 0000 0000 a84a 0e00  1|9999.......J..
00000110: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000120: 437c 3131 317c 4141 4141 0a00 0000 0000  C|111|AAAA......
00000130: c861 0c00 0500 0000 39ec 1700 042e 7b99  .a......9.....{.
00000140: ffff ffff 437c 3131 317c 4242 4242 0a00  ....C|111|BBBB..
00000150: 0000 0000 2837 0c00 0500 0000 39ec 1700  ....(7......9...
00000160: 042e 7b99 ffff ffff 437c 3131 317c 4343  ..{.....C|111|CC
00000170: 4343 0a00 0000 0000 283b 0c00 0500 0000  CC......(;......
00000180: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000190: 317c 4444 4444 0a00 0000 0000 2838 0c00  1|DDDD......(8..
000001a0: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
000001b0: 437c 3131 317c 4545 4545 0a00 0000 0000  C|111|EEEE......
000001c0: a830 0c00 0500 0000 39ec 1700 042e 7b99  .0......9.....{.
000001d0: ffff ffff 437c 3131 317c 4646 4646 0a00  ....C|111|FFFF..
000001e0: 0000 0000 08f7 1200 0500 0000 39ec 1700  ............9...
000001f0: 042e 7b99 ffff ffff 437c 3131 317c 3030  ..{.....C|111|00
00000200: 3030 0a00 0000 0000 e83f 0c00 0500 0000  00.......?......
00000210: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000220: 317c 3131 3131 0a00 0000 0000 6845 0c00  1|1111......hE..
00000230: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000240: 437c 3131 317c 3232 3232 0a00 0000 0000  C|111|2222......
00000250: 48c6 0c00 0500 0000 39ec 1700 042e 7b99  H.......9.....{.
00000260: ffff ffff 437c 3131 317c 3333 3333 0a00  ....C|111|3333..
00000270: 0000 0000 4843 0c00 0500 0000 39ec 1700  ....HC......9...
00000280: 042e 7b99 ffff ffff 437c 3131 317c 3434  ..{.....C|111|44
00000290: 3434 0a00 0000 0000 6899 0c00 0500 0000  44......h.......
000002a0: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
000002b0: 317c 3535 3535 0a00 0000 0000 2841 0c00  1|5555......(A..
000002c0: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
000002d0: 437c 3131 317c 3636 3636 0a00 0000 0000  C|111|6666......
000002e0: c83a 0c00 0500 0000 39ec 1700 042e 7b99  .:......9.....{.
000002f0: ffff ffff 437c 3131 317c 3737 3737 0a00  ....C|111|7777..
00000300: 0000 0000 c836 0c00 0500 0000 39ec 1700  .....6......9...
00000310: 042e 7b99 ffff ffff 437c 3131 317c 3838  ..{.....C|111|88
00000320: 3838 0a00 0000 0000 e82a 0c00 0500 0000  88.......*......
00000330: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000340: 317c 3939 3939 0a00 0000 0000 0834 0c00  1|9999.......4..
00000350: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000360: 437c 3131 317c 4141 4141 0a00 0000 0000  C|111|AAAA......
00000370: e883 0c00 0500 0000 39ec 1700 042e 7b99  ........9.....{.
00000380: ffff ffff 437c 3131 317c 4242 4242 0a00  ....C|111|BBBB..
00000390: 0000 0000 684f 0c00 0500 0000 39ec 1700  ....hO......9...
000003a0: 042e 7b99 ffff ffff 437c 3131 317c 4343  ..{.....C|111|CC
000003b0: 4343 0a00 0000 0000 c83a 0c00 0500 0000  CC.......:......
000003c0: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
000003d0: 317c 4444 4444 0a00 0000 0000 8836 0c00  1|DDDD.......6..
000003e0: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
000003f0: 437c 3131 317c 4545 4545 0a00 0000 0000  C|111|EEEE......
00000400: c826 0c00 0500 0000 39ec 1700 042e 7b99  .&......9.....{.
00000410: ffff ffff 437c 3131 317c 4646 4646 0a00  ....C|111|FFFF..
00000420: 0000 0000 68ee 1200 0500 0000 39ec 1700  ....h.......9...
00000430: 042e 7b99 ffff ffff 437c 3131 317c 3030  ..{.....C|111|00
00000440: 3030 0a00 0000 0000 4843 0c00 0500 0000  00......HC......
00000450: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000460: 317c 3131 3131 0a00 0000 0000 883f 0c00  1|1111.......?..
00000470: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000480: 437c 3131 317c 3232 3232 0a00 0000 0000  C|111|2222......
00000490: a824 0c00 0500 0000 39ec 1700 042e 7b99  .$......9.....{.
000004a0: ffff ffff 437c 3131 317c 3333 3333 0a00  ....C|111|3333..
000004b0: 0000 0000 e888 0d00 0500 0000 39ec 1700  ............9...
000004c0: 042e 7b99 ffff ffff 437c 3131 317c 3434  ..{.....C|111|44
000004d0: 3434 0a00 0000 0000 4853 0c00 0500 0000  44......HS......
000004e0: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
000004f0: 317c 3535 3535 0a00 0000 0000 683f 0c00  1|5555......h?..
00000500: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000510: 437c 3131 317c 3636 3636 0a00 0000 0000  C|111|6666......
00000520: 2839 0c00 0500 0000 39ec 1700 042e 7b99  (9......9.....{.
00000530: ffff ffff 437c 3131 317c 3737 3737 0a00  ....C|111|7777..
00000540: 0000 0000 4869 0c00 0500 0000 39ec 1700  ....Hi......9...
00000550: 042e 7b99 ffff ffff 437c 3131 317c 3838  ..{.....C|111|88
00000560: 3838 0a00 0000 0000 e861 0c00 0500 0000  88.......a......
00000570: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000580: 317c 3939 3939 0a00 0000 0000 4840 0c00  1|9999......H@..
00000590: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
000005a0: 437c 3131 317c 4141 4141 0a00 0000 0000  C|111|AAAA......
000005b0: a84c 0c00 0500 0000 39ec 1700 042e 7b99  .L......9.....{.
000005c0: ffff ffff 437c 3131 317c 4242 4242 0a00  ....C|111|BBBB..
000005d0: 0000 0000 2837 0c00 0500 0000 39ec 1700  ....(7......9...
000005e0: 042e 7b99 ffff ffff 437c 3131 317c 4343  ..{.....C|111|CC
000005f0: 4343 0a00 0000 0000 c831 0c00 0500 0000  CC.......1......
00000600: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000610: 317c 4444 4444 0a00 0000 0000 a823 0c00  1|DDDD.......#..
00000620: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000630: 437c 3131 317c 4545 4545 0a00 0000 0000  C|111|EEEE......
00000640: 8824 0c00 0500 0000 39ec 1700 042e 7b99  .$......9.....{.
00000650: ffff ffff 437c 3131 317c 4646 4646 0a00  ....C|111|FFFF..
00000660: 0000 0000 6867 1500 0500 0000 39ec 1700  ....hg......9...
00000670: 042e 7b99 ffff ffff 437c 3131 317c 3030  ..{.....C|111|00
00000680: 3030 0a00 0000 0000 284e 0c00 0500 0000  00......(N......
00000690: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
000006a0: 317c 3131 3131 0a00 0000 0000 8847 0c00  1|1111.......G..
000006b0: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
000006c0: 437c 3131 317c 3232 3232 0a00 0000 0000  C|111|2222......
000006d0: 6835 0c00 0500 0000 39ec 1700 042e 7b99  h5......9.....{.
000006e0: ffff ffff 437c 3131 317c 3333 3333 0a00  ....C|111|3333..
000006f0: 0000 0000 084e 0c00 0500 0000 39ec 1700  .....N......9...
00000700: 042e 7b99 ffff ffff 437c 3131 317c 3434  ..{.....C|111|44
00000710: 3434 0a00 0000 0000 c844 0c00 0500 0000  44.......D......
00000720: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000730: 317c 3535 3535 0a00 0000 0000 6825 0c00  1|5555......h%..
00000740: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000750: 437c 3131 317c 3636 3636 0a00 0000 0000  C|111|6666......
00000760: 282e 0c00 0500 0000 39ec 1700 042e 7b99  (.......9.....{.
00000770: ffff ffff 437c 3131 317c 3737 3737 0a00  ....C|111|7777..
00000780: 0000 0000 8832 0c00 0500 0000 39ec 1700  .....2......9...
00000790: 042e 7b99 ffff ffff 437c 3131 317c 3838  ..{.....C|111|88
000007a0: 3838 0a00 0000 0000 0831 0c00 0500 0000  88.......1......
000007b0: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
000007c0: 317c 3939 3939 0a00 0000 0000 e83b 0c00  1|9999.......;..
000007d0: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
000007e0: 437c 3131 317c 4141 4141 0a00 0000 0000  C|111|AAAA......
000007f0: e842 0c00 0500 0000 39ec 1700 042e 7b99  .B......9.....{.
00000800: ffff ffff 437c 3131 317c 4242 4242 0a00  ....C|111|BBBB..
00000810: 0000 0000 682d 0c00 0500 0000 39ec 1700  ....h-......9...
00000820: 042e 7b99 ffff ffff 437c 3131 317c 4343  ..{.....C|111|CC
00000830: 4343 0a00 0000 0000 4834 0c00 0500 0000  CC......H4......
00000840: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000850: 317c 4444 4444 0a00 0000 0000 887f 0c00  1|DDDD..........
00000860: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000870: 437c 3131 317c 4545 4545 0a00 0000 0000  C|111|EEEE......
00000880: 0848 0c00 0500 0000 39ec 1700 042e 7b99  .H......9.....{.
00000890: ffff ffff 437c 3131 317c 4646 4646 0a00  ....C|111|FFFF..
000008a0: 0000 0000 680e 1300 0500 0000 39ec 1700  ....h.......9...
000008b0: 042e 7b99 ffff ffff 437c 3131 317c 3030  ..{.....C|111|00
000008c0: 3030 0a00 0000 0000 e847 0c00 0500 0000  00.......G......
000008d0: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
000008e0: 317c 3131 3131 0a00 0000 0000 684f 0c00  1|1111......hO..
000008f0: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000900: 437c 3131 317c 3232 3232 0a00 0000 0000  C|111|2222......
00000910: 4844 0c00 0500 0000 39ec 1700 042e 7b99  HD......9.....{.
00000920: ffff ffff 437c 3131 317c 3333 3333 0a00  ....C|111|3333..
00000930: 0000 0000 e83e 0c00 0500 0000 39ec 1700  .....>......9...
00000940: 042e 7b99 ffff ffff 437c 3131 317c 3434  ..{.....C|111|44
00000950: 3434 0a00 0000 0000 882c 0c00 0500 0000  44.......,......
00000960: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000970: 317c 3535 3535 0a00 0000 0000 4825 0c00  1|5555......H%..
00000980: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000990: 437c 3131 317c 3636 3636 0a00 0000 0000  C|111|6666......
000009a0: 0827 0c00 0500 0000 39ec 1700 042e 7b99  .'......9.....{.
000009b0: ffff ffff 437c 3131 317c 3737 3737 0a00  ....C|111|7777..
000009c0: 0000 0000 882f 0c00 0500 0000 39ec 1700  ...../......9...
000009d0: 042e 7b99 ffff ffff 437c 3131 317c 3838  ..{.....C|111|88
000009e0: 3838 0a00 0000 0000 e835 0c00 0500 0000  88.......5......
000009f0: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000a00: 317c 3939 3939 0a00 0000 0000 48e1 0c00  1|9999......H...
00000a10: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000a20: 437c 3131 317c 4141 4141 0a00 0000 0000  C|111|AAAA......
00000a30: e85a 0c00 0500 0000 39ec 1700 042e 7b99  .Z......9.....{.
00000a40: ffff ffff 437c 3131 317c 4242 4242 0a00  ....C|111|BBBB..
00000a50: 0000 0000 2830 0c00 0500 0000 39ec 1700  ....(0......9...
00000a60: 042e 7b99 ffff ffff 437c 3131 317c 4343  ..{.....C|111|CC
00000a70: 4343 0a00 0000 0000 c833 0c00 0500 0000  CC.......3......
00000a80: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000a90: 317c 4444 4444 0a00 0000 0000 c838 0c00  1|DDDD.......8..
00000aa0: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000ab0: 437c 3131 317c 4545 4545 0a00 0000 0000  C|111|EEEE......
00000ac0: c82a 0c00 0500 0000 39ec 1700 042e 7b99  .*......9.....{.
00000ad0: ffff ffff 437c 3131 317c 4646 4646 0a00  ....C|111|FFFF..
00000ae0: 0000 0000 08e8 1200 0500 0000 39ec 1700  ............9...
00000af0: 042e 7b99 ffff ffff 437c 3131 317c 3030  ..{.....C|111|00
00000b00: 3030 0a00 0000 0000 283b 0c00 0500 0000  00......(;......
00000b10: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000b20: 317c 3131 3131 0a00 0000 0000 882d 0c00  1|1111.......-..
00000b30: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000b40: 437c 3131 317c 3232 3232 0a00 0000 0000  C|111|2222......
00000b50: 8842 0c00 0500 0000 39ec 1700 042e 7b99  .B......9.....{.
00000b60: ffff ffff 437c 3131 317c 3333 3333 0a00  ....C|111|3333..
00000b70: 0000 0000 e88e 0c00 0500 0000 39ec 1700  ............9...
00000b80: 042e 7b99 ffff ffff 437c 3131 317c 3434  ..{.....C|111|44
00000b90: 3434 0a00 0000 0000 c848 0c00 0500 0000  44.......H......
00000ba0: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000bb0: 317c 3535 3535 0a00 0000 0000 2840 0e00  1|5555......(@..
00000bc0: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000bd0: 437c 3131 317c 3636 3636 0a00 0000 0000  C|111|6666......
00000be0: a878 0c00 0500 0000 39ec 1700 042e 7b99  .x......9.....{.
00000bf0: ffff ffff 437c 3131 317c 3737 3737 0a00  ....C|111|7777..
00000c00: 0000 0000 8837 0c00 0500 0000 39ec 1700  .....7......9...
00000c10: 042e 7b99 ffff ffff 437c 3131 317c 3838  ..{.....C|111|88
00000c20: 3838 0a00 0000 0000 482e 0c00 0500 0000  88......H.......
00000c30: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000c40: 317c 3939 3939 0a00 0000 0000 6832 0c00  1|9999......h2..
00000c50: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000c60: 437c 3131 317c 4141 4141 0a00 0000 0000  C|111|AAAA......
00000c70: 483a 0c00 0500 0000 39ec 1700 042e 7b99  H:......9.....{.
00000c80: ffff ffff 437c 3131 317c 4242 4242 0a00  ....C|111|BBBB..
00000c90: 0000 0000 4830 0c00 0500 0000 39ec 1700  ....H0......9...
00000ca0: 042e 7b99 ffff ffff 437c 3131 317c 4343  ..{.....C|111|CC
00000cb0: 4343 0a00 0000 0000 282e 0c00 0500 0000  CC......(.......
00000cc0: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000cd0: 317c 4444 4444 0a00 0000 0000 c81f 1c00  1|DDDD..........
00000ce0: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000cf0: 437c 3131 317c 4545 4545 0a00 0000 0000  C|111|EEEE......
00000d00: 6833 0c00 0500 0000 39ec 1700 042e 7b99  h3......9.....{.
00000d10: ffff ffff 437c 3131 317c 4646 4646 0a00  ....C|111|FFFF..
00000d20: 0000 0000 28ee 1200 0500 0000 39ec 1700  ....(.......9...
00000d30: 042e 7b99 ffff ffff 437c 3131 317c 3030  ..{.....C|111|00
00000d40: 3030 0a00 0000 0000 083e 0c00 0500 0000  00.......>......
00000d50: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000d60: 317c 3131 3131 0a00 0000 0000 8848 0c00  1|1111.......H..
00000d70: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000d80: 437c 3131 317c 3232 3232 0a00 0000 0000  C|111|2222......
00000d90: 2832 0c00 0500 0000 39ec 1700 042e 7b99  (2......9.....{.
00000da0: ffff ffff 437c 3131 317c 3333 3333 0a00  ....C|111|3333..
00000db0: 0000 0000 e8b9 1400 0500 0000 39ec 1700  ............9...
00000dc0: 042e 7b99 ffff ffff 437c 3131 317c 3434  ..{.....C|111|44
00000dd0: 3434 0a00 0000 0000 28ff 0c00 0500 0000  44......(.......
00000de0: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000df0: 317c 3535 3535 0a00 0000 0000 08f3 0c00  1|5555..........
00000e00: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000e10: 437c 3131 317c 3636 3636 0a00 0000 0000  C|111|6666......
00000e20: 88f0 0c00 0500 0000 39ec 1700 042e 7b99  ........9.....{.
00000e30: ffff ffff 437c 3131 317c 3737 3737 0a00  ....C|111|7777..
00000e40: 0000 0000 88e8 1300 0500 0000 39ec 1700  ............9...
00000e50: 042e 7b99 ffff ffff 437c 3131 317c 3838  ..{.....C|111|88
00000e60: 3838 0a00 0000 0000 e8c0 1600 0500 0000  88..............
00000e70: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000e80: 317c 3939 3939 0a00 0000 0000 c878 0e00  1|9999.......x..
00000e90: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000ea0: 437c 3131 317c 4141 4141 0a00 0000 0000  C|111|AAAA......
00000eb0: 0889 0e00 0500 0000 39ec 1700 042e 7b99  ........9.....{.
00000ec0: ffff ffff 437c 3131 317c 4242 4242 0a00  ....C|111|BBBB..
00000ed0: 0000 0000 286f 0e00 0500 0000 39ec 1700  ....(o......9...
00000ee0: 042e 7b99 ffff ffff 437c 3131 317c 4343  ..{.....C|111|CC
00000ef0: 4343 0a00 0000 0000 4874 0e00 0500 0000  CC......Ht......
00000f00: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000f10: 317c 4444 4444 0a00 0000 0000 8869 0e00  1|DDDD.......i..
00000f20: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000f30: 437c 3131 317c 4545 4545 0a00 0000 0000  C|111|EEEE......
00000f40: 8870 0e00 0500 0000 39ec 1700 042e 7b99  .p......9.....{.
00000f50: ffff ffff 437c 3131 317c 4646 4646 0a00  ....C|111|FFFF..
00000f60: 0000 0000 6892 1600 0500 0000 39ec 1700  ....h.......9...
00000f70: 042e 7b99 ffff ffff 437c 3131 317c 3030  ..{.....C|111|00
00000f80: 3030 0a00 0000 0000 e82e 0f00 0500 0000  00..............
00000f90: 39ec 1700 042e 7b99 ffff ffff 437c 3131  9.....{.....C|11
00000fa0: 317c 3131 3131 0a00 0000 0000 08ab 0e00  1|1111..........
00000fb0: 0500 0000 39ec 1700 042e 7b99 ffff ffff  ....9.....{.....
00000fc0: 437c 3131 317c 3232 3232 0a00 0000 0000  C|111|2222......
00000fd0: 2885 0e00 0500 0000 39ec 1700 042e 7b99  (.......9.....{.
00000fe0: ffff ffff 437c 3131 317c 3333 3333 0a00  ....C|111|3333..
00000ff0: 0000 0000 5301 0000 0000 0000 0000 0000  ....S...........
    )",
};

FtraceDataSourceConfig ConfigForTesting(CompactSchedConfig compact_cfg) {
  return FtraceDataSourceConfig{
      /*event_filter=*/EventFilter{},
      /*syscall_filter=*/EventFilter{}, compact_cfg,
      /*print_filter=*/std::nullopt,
      /*atrace_apps=*/{},
      /*atrace_categories=*/{},
      /*atrace_categories_sdk_optout=*/{},
      /*symbolize_ksyms=*/false,
      /*buffer_percent=*/0u,
      /*syscalls_returning_fd=*/{},
      /*kprobes=*/
      base::FlatHashMap<uint32_t, protos::pbzero::KprobeEvent::KprobeType>{0},
      /*debug_ftrace_abi=*/false,
      /*write_generic_evt_descriptors=*/false};
}

// Low level benchmark for the CpuReader::ParsePageHeader and
// CpuReader::ParsePagePayload functions.
void DoParse(const ExamplePage& test_case,
             const std::vector<GroupAndName>& enabled_events,
             std::optional<FtraceConfig::PrintFilter> print_filter,
             benchmark::State& state) {
  NullTraceWriter writer;
  FtraceMetadata metadata{};
  auto compact_sched_buf = std::make_unique<CompactSchedBuffer>();
  GenericEventProtoDescriptors generic_pb_descriptors;
  CpuReader::Bundler bundler(
      &writer, &metadata, /*symbolizer=*/nullptr, /*cpu=*/0,
      /*clock_snapshot=*/std::nullopt, compact_sched_buf.get(),
      /*compact_sched_enabled=*/false, /*previous_bundle_end_ts=*/0,
      &generic_pb_descriptors);

  ProtoTranslationTable* table = GetTable(test_case.name);
  auto page = PageFromXxd(test_case.data);

  FtraceDataSourceConfig ds_config =
      ConfigForTesting(DisabledCompactSchedConfigForTesting());
  if (print_filter.has_value()) {
    ds_config.print_filter =
        FtracePrintFilterConfig::Create(print_filter.value(), table);
    PERFETTO_CHECK(ds_config.print_filter.has_value());
  }
  for (const GroupAndName& enabled_event : enabled_events) {
    ds_config.event_filter.AddEnabledEvent(
        table->EventToFtraceId(enabled_event));
  }

  while (state.KeepRunning()) {
    std::unique_ptr<CompactSchedBuffer> compact_buffer(
        new CompactSchedBuffer());
    const uint8_t* parse_pos = page.get();
    std::optional<CpuReader::PageHeader> page_header =
        CpuReader::ParsePageHeader(&parse_pos, table->page_header_size_len());

    if (!page_header.has_value())
      return;

    uint64_t last_read_event_ts = 0;
    CpuReader::ParsePagePayload(parse_pos, &page_header.value(), table,
                                &ds_config, &bundler, &metadata,
                                &last_read_event_ts);

    metadata.Clear();
    bundler.FinalizeAndRunSymbolizer();
  }
}

void BM_ParsePageFullOfSchedSwitch(benchmark::State& state) {
  DoParse(g_full_page_sched_switch, {GroupAndName("sched", "sched_switch")},
          std::nullopt, state);
}
BENCHMARK(BM_ParsePageFullOfSchedSwitch);

void BM_ParsePageFullOfPrint(benchmark::State& state) {
  DoParse(g_full_page_print, {GroupAndName("ftrace", "print")}, std::nullopt,
          state);
}
BENCHMARK(BM_ParsePageFullOfPrint);

void BM_ParsePageFullOfPrintWithFilterRules(benchmark::State& state) {
  FtraceConfig::PrintFilter filter_conf;
  for (int i = 0; i < state.range(0); i++) {
    auto* rule = filter_conf.add_rules();
    rule->set_prefix("X");  // This rule will not match
    rule->set_allow(false);
  }
  DoParse(g_full_page_print, {GroupAndName("ftrace", "print")}, filter_conf,
          state);
}
BENCHMARK(BM_ParsePageFullOfPrintWithFilterRules)->DenseRange(0, 16, 1);

void BM_ParsePageFullOfAtracePrint(benchmark::State& state) {
  DoParse(g_full_page_atrace_print, {GroupAndName("ftrace", "print")},
          std::nullopt, state);
}
BENCHMARK(BM_ParsePageFullOfAtracePrint);

void BM_ParsePageFullOfAtracePrintWithFilterRules(benchmark::State& state) {
  FtraceConfig::PrintFilter filter_conf;
  for (int i = 0; i < state.range(0); i++) {
    auto* rule = filter_conf.add_rules();
    auto* atrace = rule->mutable_atrace_msg();
    atrace->set_type("C");
    atrace->set_prefix("X");  // This rule will not match
    rule->set_allow(false);
  }
  DoParse(g_full_page_print, {GroupAndName("ftrace", "print")}, filter_conf,
          state);
}
BENCHMARK(BM_ParsePageFullOfAtracePrintWithFilterRules)->DenseRange(0, 16, 1);

// Higher level benchmark for the CpuReader::ProcessPagesForDataSource function.
void DoProcessPages(const ExamplePage& test_case,
                    const size_t page_repetition,
                    const std::vector<GroupAndName>& enabled_events,
                    std::optional<FtraceConfig::PrintFilter> print_filter,
                    benchmark::State& state) {
  perfetto::NullTraceWriter writer;

  ProtoTranslationTable* table = GetTable(test_case.name);

  auto repeated_pages =
      std::make_unique<uint8_t[]>(base::GetSysPageSize() * page_repetition);
  {
    auto page = PageFromXxd(test_case.data);
    for (size_t i = 0; i < page_repetition; i++) {
      memcpy(&repeated_pages[i * base::GetSysPageSize()], &page[0],
             base::GetSysPageSize());
    }
  }

  FtraceDataSourceConfig ds_config =
      ConfigForTesting(DisabledCompactSchedConfigForTesting());
  if (print_filter.has_value()) {
    ds_config.print_filter =
        FtracePrintFilterConfig::Create(print_filter.value(), table);
    PERFETTO_CHECK(ds_config.print_filter.has_value());
  }
  for (const GroupAndName& enabled_event : enabled_events) {
    ds_config.event_filter.AddEnabledEvent(
        table->EventToFtraceId(enabled_event));
  }

  FtraceMetadata metadata{};
  auto compact_sched_buf = std::make_unique<CompactSchedBuffer>();
  uint64_t last_read_event_ts = 0;
  base::FlatSet<protos::pbzero::FtraceParseStatus> parse_errors;
  while (state.KeepRunning()) {
    CpuReader::ProcessPagesForDataSource(
        &writer, &metadata, /*cpu=*/0, &ds_config, &parse_errors,
        &last_read_event_ts, repeated_pages.get(), page_repetition,
        compact_sched_buf.get(), table,
        /*symbolizer=*/nullptr, /*clock_snapshot=*/std::nullopt);

    metadata.Clear();
  }
}

void BM_ProcessPagesFullOfSchedSwitch(benchmark::State& state) {
  DoProcessPages(g_full_page_sched_switch, static_cast<size_t>(state.range(0)),
                 {GroupAndName("sched", "sched_switch")}, std::nullopt, state);
}
BENCHMARK(BM_ProcessPagesFullOfSchedSwitch)->Range(1, 64);

void BM_ProcessPagesFullOfPrint(benchmark::State& state) {
  DoProcessPages(g_full_page_print, static_cast<size_t>(state.range(0)),
                 {GroupAndName("ftrace", "print")}, std::nullopt, state);
}
BENCHMARK(BM_ProcessPagesFullOfPrint)->Range(1, 64);

}  // namespace
}  // namespace perfetto
